from abc import abstractmethod
from typing import List, Literal, Optional

import yaml
from pydantic import BaseModel, Field


class BaseConfig:

    @abstractmethod
    def load(self, file_path: str) -> None:
        pass


class LLMPromptSetting(BaseModel):

    validate_spill_prompt: str = Field(
        ...,
        description="spill prompt for judge task",
        alias="judge_spill_prompt",
    )

    reco_spill_prompt: str = Field(
        ...,
        description="spill prompt for reco task",
        alias="reco_spill_prompt",
    )

    redraw_spill_prompt: str = Field(
        ...,
        description="spill prompt for redraw task",
        alias="redraw_spill_prompt",
    )


class WebAPISetting(BaseModel):
    host: str = Field(..., description="webapi host", alias="host")
    port: int = Field(..., description="webapi port", alias="port")


class ImageCacheSetting(BaseModel):
    source_path: str = Field(..., description="image cache path", alias="source_path")
    target_path: str = Field(..., description="image cache path", alias="target_path")


class OllamaClientSetting(BaseModel):
    host: str = Field(..., description="ollama client host", alias="host")
    port: int = Field(..., description="ollama client port", alias="port")


class JobsSettings(BaseModel):
    name: str = Field(..., description="job name", alias="name")
    a_type: Literal["chain", "signal"] = Field(
        ..., description="job type", alias="type"
    )


class ModelSetting(BaseModel):
    name: str = Field(..., description="model name", alias="name")
    path: str = Field(..., description="model path", alias="path")


class RootSetting(BaseModel):
    llm_prompt: LLMPromptSetting
    webapi: WebAPISetting
    image_cache: ImageCacheSetting
    ollama_client: OllamaClientSetting
    jobs: List[JobsSettings]
    models: List[ModelSetting]


class YamlConfig(BaseConfig):

    def load(self, file_path: str) -> Optional[RootSetting]:
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                inst_config = yaml.safe_load(f)
                settings = RootSetting(**inst_config)
                return settings
        except Exception as e:
            raise e
