import os
import pathlib
import aiofiles
import uvicorn
from fastapi import FastAPI
from pydantic import BaseModel, Field
from helps.logger import AppLogger
from helps.config import RootSetting


app = FastAPI()
logger = AppLogger.get_logger(__name__)
image_cache_ppath = "./caches/images/"


class RecoPostModel(BaseModel):
    image_uuid: str = Field(..., description="image uuid", alias="image_uuid")
    image_b64: str = Field(..., description="image b64", alias="image_b64")


async def convert_str_to_b64(str_b64: str) -> bytes:
    return bytes(str_b64, "utf-8")


@app.post("/beeble/reco")
async def reco(post_body: RecoPostModel):
    logger.info(post_body.image_uuid)
    full_path = pathlib.Path(image_cache_ppath) / post_body.image_uuid
    content = await convert_str_to_b64(post_body.image_b64)

    os.makedirs(image_cache_ppath, exist_ok=True)

    try:
        async with aiofiles.open(full_path, "wb") as f:
            await f.write(content)

    except IOError as err:
        logger.error(str(err))
        return {"status": "failed", "message": f"failed {str(err)}"}

    return {"status": "success", "message": "complete"}


@app.get("/beeble/health")
async def health():
    return {"message": "Hello World"}


async def start(settings: RootSetting):
    config = uvicorn.Config(
        app=app,
        host=settings.webapi.host,
        port=settings.webapi.port,
        log_level=logger.level,
        reload=True,
    )
    global image_cache_ppath
    image_cache_ppath = settings.image_cache.p_path
    server = uvicorn.Server(config)
    await server.serve()
