import cv2


def compare_images(img1_path, img2_path, threshold=30):
    # Load images
    img1 = cv2.imread(img1_path)
    img2 = cv2.imread(img2_path)

    # Resize if needed
    if img1.shape != img2.shape:
        img2 = cv2.resize(img2, (img1.shape[1], img1.shape[0]))

    # Convert to grayscale
    gray1 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
    gray2 = cv2.cvtColor(img2, cv2.COLOR_BGR2GRAY)

    # Find difference
    diff = cv2.absdiff(gray1, gray2)
    _, mask = cv2.threshold(diff, threshold, 255, cv2.THRESH_BINARY)

    # Find contours
    contours, _ = cv2.findContours(mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)

    # Draw differences
    result = img1.copy()
    for contour in contours:
        if cv2.contourArea(contour) > 100:
            x, y, w, h = cv2.boundingRect(contour)
            cv2.rectangle(result, (x, y), (x + w, y + h), (0, 0, 255), 2)

    return result, mask, len([c for c in contours if cv2.contourArea(c) > 100])


# Usage - Save images instead of showing
result, mask, num_differences = compare_images(
    "images/image_1.jpg", "images/image_2.jpg"
)
print(f"Found {num_differences} differences")

# Save the result images
cv2.imwrite(
    "differences_highlighted.jpg", result
)  # Image with red rectangles around differences
cv2.imwrite("differences_mask.jpg", mask)  # Binary mask showing differences in white

print("Images saved:")
print("- differences_highlighted.jpg (original image with differences marked)")
print("- differences_mask.jpg (binary mask of differences)")
