import cv2
import numpy as np
from sklearn.cluster import KMeans
import matplotlib.pyplot as plt
from skimage import segmentation, measure
import os


class EnhancedBuildingChangeDetector:
    def __init__(self):
        """
        Enhanced Building-Focused Change Detection System
        Improved shadow removal, vegetation filtering, and building shape analysis
        """
        self.kernel_size = 7
        self.morph_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (7, 7))

    def preprocess_image(self, image):
        """
        Enhanced image preprocessing with shadow correction
        """
        # Gaussian filtering for noise reduction
        denoised = cv2.GaussianBlur(image, (5, 5), 0)

        # Shadow correction using adaptive histogram equalization
        if len(image.shape) == 3:
            # Convert to LAB color space for better shadow handling
            lab = cv2.cvtColor(denoised, cv2.COLOR_BGR2LAB)

            # Apply CLAHE to L channel to reduce shadow effects
            clahe = cv2.createCLAHE(clipLimit=3.0, tileGridSize=(8, 8))
            lab[:, :, 0] = clahe.apply(lab[:, :, 0])
            enhanced = cv2.cvtColor(lab, cv2.COLOR_LAB2BGR)
        else:
            clahe = cv2.createCLAHE(clipLimit=3.0, tileGridSize=(8, 8))
            enhanced = clahe.apply(denoised)

        return enhanced

    def detect_and_remove_shadows(self, image):
        """
        Advanced shadow detection and removal
        """
        # Convert to multiple color spaces for comprehensive shadow detection
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)
        yuv = cv2.cvtColor(image, cv2.COLOR_BGR2YUV)

        shadow_mask = np.zeros(image.shape[:2], dtype=np.uint8)

        # Method 1: Low brightness detection in HSV
        v_channel = hsv[:, :, 2]
        low_brightness = v_channel < 80
        shadow_mask1 = np.uint8(low_brightness * 255)

        # Method 2: Blue dominance (shadows often have blue cast)
        b, g, r = cv2.split(image)
        blue_dominant = (b > g) & (b > r) & (b > 60)
        shadow_mask2 = np.uint8(blue_dominant * 255)

        # Method 3: Low saturation + low value
        s_channel = hsv[:, :, 1]
        low_sat_val = (s_channel < 50) & (v_channel < 100)
        shadow_mask3 = np.uint8(low_sat_val * 255)

        # Method 4: Using LAB color space - shadows have specific characteristics
        a_channel = lab[:, :, 1]
        b_channel = lab[:, :, 2]
        l_channel = lab[:, :, 0]

        # Shadows typically have low L and shifted A/B values
        shadow_lab = (l_channel < 80) & (a_channel > 120) & (b_channel < 135)
        shadow_mask4 = np.uint8(shadow_lab * 255)

        # Combine all shadow detection methods
        shadow_mask = cv2.bitwise_or(shadow_mask1, shadow_mask2)
        shadow_mask = cv2.bitwise_or(shadow_mask, shadow_mask3)
        shadow_mask = cv2.bitwise_or(shadow_mask, shadow_mask4)

        # Morphological operations to clean up shadow mask
        shadow_kernel = cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (15, 15))
        shadow_mask = cv2.morphologyEx(shadow_mask, cv2.MORPH_CLOSE, shadow_kernel)
        shadow_mask = cv2.morphologyEx(
            shadow_mask,
            cv2.MORPH_OPEN,
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (8, 8)),
        )

        # Remove shadows from image
        result = image.copy()

        # Instead of setting to gray, try to brighten shadow areas
        shadow_indices = shadow_mask > 0
        if np.any(shadow_indices):
            # Brighten shadows by increasing brightness
            hsv_temp = cv2.cvtColor(result, cv2.COLOR_BGR2HSV)
            hsv_temp[shadow_indices, 2] = np.minimum(
                255, hsv_temp[shadow_indices, 2] * 1.5
            )
            result = cv2.cvtColor(hsv_temp, cv2.COLOR_HSV2BGR)

        return result, shadow_mask

    def remove_vegetation_enhanced(self, image):
        """
        Enhanced vegetation removal with multiple detection methods
        """
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)

        vegetation_mask = np.zeros(hsv.shape[:2], dtype=np.uint8)

        # Method 1: Enhanced green detection in HSV with multiple ranges
        green_ranges = [
            ([25, 20, 15], [95, 255, 240]),  # Very broad green
            ([30, 30, 20], [85, 255, 255]),  # Standard green
            ([35, 40, 25], [80, 200, 200]),  # Dark green (shadows)
            ([40, 60, 40], [75, 255, 255]),  # Bright green (sunlit)
            ([45, 80, 60], [70, 255, 180]),  # Specific tree green
        ]

        for lower, upper in green_ranges:
            mask = cv2.inRange(hsv, np.array(lower), np.array(upper))
            vegetation_mask = cv2.bitwise_or(vegetation_mask, mask)

        # Method 2: Enhanced LAB color space detection
        a_channel = lab[:, :, 1]
        b_channel = lab[:, :, 2]
        l_channel = lab[:, :, 0]

        # Green areas have low 'a' values and specific 'b' characteristics
        green_lab_mask = (a_channel < 125) & (b_channel > 120) & (l_channel > 30)
        green_lab_mask = np.uint8(green_lab_mask * 255)

        # Method 3: NDVI-like index (Normalized Difference Vegetation Index)
        b_bgr, g_bgr, r_bgr = cv2.split(image)

        # Avoid division by zero
        denominator = r_bgr.astype(np.float32) + g_bgr.astype(np.float32)
        denominator[denominator == 0] = 1

        ndvi_like = (g_bgr.astype(np.float32) - r_bgr.astype(np.float32)) / denominator
        ndvi_mask = (ndvi_like > 0.1) & (g_bgr > 50)  # Vegetation has positive NDVI
        ndvi_mask = np.uint8(ndvi_mask * 255)

        # Method 4: Texture-based detection for organic patterns
        gray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)

        # Use Gabor filters to detect organic textures
        def apply_gabor_filter(img, theta):
            kernel = cv2.getGaborKernel(
                (21, 21), 5, theta, 10, 0.5, 0, ktype=cv2.CV_32F
            )
            return cv2.filter2D(img, cv2.CV_8UC3, kernel)

        # Apply multiple Gabor filters for different orientations
        gabor_responses = []
        for angle in [0, 45, 90, 135]:
            gabor_resp = apply_gabor_filter(gray, np.radians(angle))
            gabor_responses.append(gabor_resp)

        # Combine Gabor responses
        gabor_combined = np.zeros_like(gray, dtype=np.float32)
        for resp in gabor_responses:
            gabor_combined += resp.astype(np.float32)

        # Vegetation has irregular, organic texture patterns
        _, organic_texture = cv2.threshold(
            gabor_combined.astype(np.uint8), 30, 255, cv2.THRESH_BINARY
        )

        # Method 5: Edge density analysis (trees have high edge density)
        edges = cv2.Canny(gray, 50, 150)
        edge_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (15, 15))
        edge_density = cv2.morphologyEx(edges, cv2.MORPH_CLOSE, edge_kernel)

        # Combine all vegetation detection methods
        vegetation_mask = cv2.bitwise_or(vegetation_mask, green_lab_mask)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, ndvi_mask)

        # Only include organic texture areas that are also green
        green_organic = cv2.bitwise_and(vegetation_mask, organic_texture)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, green_organic)

        # Include high edge density areas that are green
        green_edges = cv2.bitwise_and(vegetation_mask, edge_density)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, green_edges)

        # Morphological operations to clean up vegetation mask
        vegetation_kernel = cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (20, 20))
        vegetation_mask = cv2.morphologyEx(
            vegetation_mask, cv2.MORPH_CLOSE, vegetation_kernel
        )
        vegetation_mask = cv2.morphologyEx(
            vegetation_mask,
            cv2.MORPH_OPEN,
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (12, 12)),
        )

        # Remove small noise
        contours, _ = cv2.findContours(
            vegetation_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )
        cleaned_mask = np.zeros_like(vegetation_mask)

        for contour in contours:
            area = cv2.contourArea(contour)
            if area > 500:  # Only keep larger vegetation areas
                cv2.fillPoly(cleaned_mask, [contour], 255)

        # Apply vegetation mask
        result = image.copy()
        result[cleaned_mask > 0] = [128, 128, 128]

        return result, cleaned_mask

    def extract_building_regions_enhanced(self, image):
        """
        Enhanced building extraction with focus on geometric shapes
        """
        gray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)

        # Method 1: Enhanced line detection for building edges
        edges = cv2.Canny(gray, 30, 150, apertureSize=3)

        # Detect both short and long lines
        lines_long = cv2.HoughLinesP(
            edges, 1, np.pi / 180, threshold=80, minLineLength=60, maxLineGap=15
        )
        lines_short = cv2.HoughLinesP(
            edges, 1, np.pi / 180, threshold=40, minLineLength=30, maxLineGap=10
        )

        line_mask = np.zeros_like(gray)

        # Draw long lines (building edges)
        if lines_long is not None:
            for line in lines_long:
                x1, y1, x2, y2 = line[0]
                cv2.line(line_mask, (x1, y1), (x2, y2), 255, 3)

        # Draw short lines (building details)
        if lines_short is not None:
            for line in lines_short:
                x1, y1, x2, y2 = line[0]
                cv2.line(line_mask, (x1, y1), (x2, y2), 128, 2)

        # Method 2: Corner detection with multiple scales
        corners_fine = cv2.cornerHarris(gray, 2, 3, 0.04)
        corners_coarse = cv2.cornerHarris(gray, 5, 7, 0.04)

        corner_mask = np.zeros_like(gray)
        corner_mask[corners_fine > 0.01 * corners_fine.max()] = 255
        corner_mask[corners_coarse > 0.01 * corners_coarse.max()] = 255

        # Method 3: Rectangular pattern detection with multiple sizes
        rect_kernels = [
            cv2.getStructuringElement(cv2.MORPH_RECT, (30, 8)),  # Large horizontal
            cv2.getStructuringElement(cv2.MORPH_RECT, (8, 30)),  # Large vertical
            cv2.getStructuringElement(cv2.MORPH_RECT, (20, 5)),  # Medium horizontal
            cv2.getStructuringElement(cv2.MORPH_RECT, (5, 20)),  # Medium vertical
            cv2.getStructuringElement(cv2.MORPH_RECT, (15, 3)),  # Small horizontal
            cv2.getStructuringElement(cv2.MORPH_RECT, (3, 15)),  # Small vertical
        ]

        rectangular_features = np.zeros_like(gray)
        for kernel in rect_kernels:
            rect_resp = cv2.morphologyEx(edges, cv2.MORPH_CLOSE, kernel)
            rectangular_features = cv2.bitwise_or(rectangular_features, rect_resp)

        # Method 4: Template matching for building-like shapes
        # Create building-like templates
        templates = []

        # Square template
        square_template = np.zeros((40, 40), dtype=np.uint8)
        cv2.rectangle(square_template, (5, 5), (35, 35), 255, 2)
        templates.append(square_template)

        # Rectangle template
        rect_template = np.zeros((30, 50), dtype=np.uint8)
        cv2.rectangle(rect_template, (5, 5), (45, 25), 255, 2)
        templates.append(rect_template)

        # L-shaped template
        l_template = np.zeros((40, 40), dtype=np.uint8)
        cv2.rectangle(l_template, (5, 5), (35, 20), 255, 2)
        cv2.rectangle(l_template, (5, 20), (20, 35), 255, 2)
        templates.append(l_template)

        template_mask = np.zeros_like(gray)
        for template in templates:
            result = cv2.matchTemplate(gray, template, cv2.TM_CCOEFF_NORMED)
            locations = np.where(result >= 0.3)
            for pt in zip(*locations[::-1]):
                cv2.rectangle(
                    template_mask,
                    pt,
                    (pt[0] + template.shape[1], pt[1] + template.shape[0]),
                    255,
                    -1,
                )

        # Method 5: Combine all building detection methods
        building_features = cv2.bitwise_or(line_mask, corner_mask)
        building_features = cv2.bitwise_or(building_features, rectangular_features)
        building_features = cv2.bitwise_or(building_features, template_mask)

        # Enhanced morphological operations for building-like shapes
        building_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (25, 25))
        building_mask = cv2.morphologyEx(
            building_features, cv2.MORPH_CLOSE, building_kernel
        )

        # Fill holes and connect nearby building parts
        contours, _ = cv2.findContours(
            building_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )
        filled_mask = np.zeros_like(building_mask)

        for contour in contours:
            area = cv2.contourArea(contour)
            if area > 2000:  # Only keep substantial building areas
                # Check if contour is building-like
                if self.is_building_like_shape(contour):
                    cv2.fillPoly(filled_mask, [contour], 255)

        return filled_mask

    def is_building_like_shape(self, contour):
        """
        Analyze if a contour represents a building-like shape
        """
        area = cv2.contourArea(contour)
        if area < 2000:
            return False

        # Get bounding rectangle
        x, y, w, h = cv2.boundingRect(contour)
        rect_area = w * h

        # Calculate shape metrics
        extent = area / rect_area if rect_area > 0 else 0
        aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

        # Calculate hull ratio (convexity)
        hull = cv2.convexHull(contour)
        hull_area = cv2.contourArea(hull)
        solidity = area / hull_area if hull_area > 0 else 0

        # Buildings should be:
        # - Reasonably rectangular (extent > 0.4)
        # - Not too elongated (aspect_ratio < 6)
        # - Fairly convex (solidity > 0.6)
        # - Have reasonable perimeter to area ratio
        perimeter = cv2.arcLength(contour, True)
        circularity = 4 * np.pi * area / (perimeter * perimeter) if perimeter > 0 else 0

        return (
            extent > 0.4
            and aspect_ratio < 6.0
            and solidity > 0.6
            and circularity > 0.1
            and area > 2000
        )

    def detect_building_changes_enhanced(
        self, img1, img2, building_mask1, building_mask2
    ):
        """
        Enhanced building change detection with roof analysis
        """
        # 1. AREA CHANGES
        area_diff = cv2.absdiff(building_mask1, building_mask2)

        # 2. ENHANCED COLOR CHANGES with roof-specific analysis
        # Buildings often have distinct roof colors/materials

        # Convert images to multiple color spaces
        hsv1 = cv2.cvtColor(img1, cv2.COLOR_BGR2HSV)
        hsv2 = cv2.cvtColor(img2, cv2.COLOR_BGR2HSV)
        lab1 = cv2.cvtColor(img1, cv2.COLOR_BGR2LAB)
        lab2 = cv2.cvtColor(img2, cv2.COLOR_BGR2LAB)

        # Enhanced HSV difference (better for roof material changes)
        hsv_diff = cv2.absdiff(hsv1, hsv2)
        hsv_change_gray = cv2.cvtColor(hsv_diff, cv2.COLOR_HSV2BGR)
        hsv_change_gray = cv2.cvtColor(hsv_change_gray, cv2.COLOR_BGR2GRAY)

        # LAB difference (better for lighting-independent changes)
        lab_diff = cv2.absdiff(lab1, lab2)
        lab_change_gray = cv2.cvtColor(lab_diff, cv2.COLOR_LAB2BGR)
        lab_change_gray = cv2.cvtColor(lab_change_gray, cv2.COLOR_BGR2GRAY)

        # Hue-specific changes (important for roof color changes)
        h1, h2 = hsv1[:, :, 0], hsv2[:, :, 0]
        hue_diff = np.abs(h1.astype(np.float32) - h2.astype(np.float32))
        # Handle hue wraparound (0 and 179 are close)
        hue_diff = np.minimum(hue_diff, 180 - hue_diff)
        hue_diff = (hue_diff * 255 / 90).astype(np.uint8)  # Normalize

        # Combine color changes with weights optimized for building detection
        color_change = cv2.addWeighted(hsv_change_gray, 0.3, lab_change_gray, 0.5, 0)
        color_change = cv2.addWeighted(color_change, 0.8, hue_diff, 0.2, 0)

        # Only consider color changes within building areas
        combined_building_mask = cv2.bitwise_or(building_mask1, building_mask2)
        building_color_change = cv2.bitwise_and(color_change, combined_building_mask)

        # 3. TEXTURE CHANGES (roof texture/material changes)
        gray1 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
        gray2 = cv2.cvtColor(img2, cv2.COLOR_BGR2GRAY)

        # Local Binary Pattern-like texture analysis
        def get_texture_features(img):
            # Use Laplacian for texture detection
            laplacian = cv2.Laplacian(img, cv2.CV_64F)
            return np.absolute(laplacian).astype(np.uint8)

        texture1 = get_texture_features(gray1)
        texture2 = get_texture_features(gray2)
        texture_diff = cv2.absdiff(texture1, texture2)

        # Only consider texture changes in building areas
        building_texture_change = cv2.bitwise_and(texture_diff, combined_building_mask)

        # 4. COMBINE ALL CHANGES with building-specific weights
        final_change = cv2.addWeighted(area_diff, 0.5, building_color_change, 0.3, 0)
        final_change = cv2.addWeighted(
            final_change, 0.8, building_texture_change, 0.2, 0
        )

        # Apply threshold for significant changes only
        _, change_mask = cv2.threshold(final_change, 40, 255, cv2.THRESH_BINARY)

        return change_mask

    def remove_roads_comprehensive(self, image):
        """
        Comprehensive road removal (from original code)
        """
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)

        road_mask = np.zeros(image.shape[:2], dtype=np.uint8)

        # Gray roads detection
        gray_ranges_bgr = [
            ([60, 60, 60], [140, 140, 140]),
            ([40, 40, 40], [100, 100, 100]),
            ([100, 100, 100], [180, 180, 180]),
            ([20, 20, 20], [70, 70, 70]),
        ]

        for lower, upper in gray_ranges_bgr:
            mask = cv2.inRange(image, np.array(lower), np.array(upper))
            road_mask = cv2.bitwise_or(road_mask, mask)

        # Low saturation areas
        low_saturation = hsv[:, :, 1] < 40
        low_sat_mask = np.uint8(low_saturation * 255)

        # Yellow line detection
        yellow_lower = np.array([20, 100, 100])
        yellow_upper = np.array([30, 255, 255])
        yellow_mask = cv2.inRange(hsv, yellow_lower, yellow_upper)
        yellow_dilated = cv2.dilate(
            yellow_mask,
            cv2.getStructuringElement(cv2.MORPH_RECT, (20, 20)),
            iterations=2,
        )

        # Combine road detection methods
        combined_road_mask = cv2.bitwise_or(road_mask, low_sat_mask)
        combined_road_mask = cv2.bitwise_or(combined_road_mask, yellow_dilated)

        # Linear structure detection
        kernels = [
            cv2.getStructuringElement(cv2.MORPH_RECT, (40, 5)),
            cv2.getStructuringElement(cv2.MORPH_RECT, (5, 40)),
        ]

        linear_structures = np.zeros_like(combined_road_mask)
        for kernel in kernels:
            linear_result = cv2.morphologyEx(combined_road_mask, cv2.MORPH_OPEN, kernel)
            linear_structures = cv2.bitwise_or(linear_structures, linear_result)

        final_road_mask = cv2.bitwise_or(combined_road_mask, linear_structures)

        # Clean up road mask
        kernel_close = cv2.getStructuringElement(cv2.MORPH_RECT, (15, 15))
        final_road_mask = cv2.morphologyEx(
            final_road_mask, cv2.MORPH_CLOSE, kernel_close
        )

        # Apply road mask
        result = image.copy()
        result[final_road_mask > 0] = [128, 128, 128]

        return result, final_road_mask

    def register_images(self, img1, img2):
        """
        Image registration (from original code)
        """
        gray1 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
        gray2 = cv2.cvtColor(img2, cv2.COLOR_BGR2GRAY)

        orb = cv2.ORB_create(nfeatures=2000)
        kp1, des1 = orb.detectAndCompute(gray1, None)
        kp2, des2 = orb.detectAndCompute(gray2, None)

        if des1 is None or des2 is None:
            print(
                "Warning: Unable to extract sufficient feature points, returning original image"
            )
            return img2

        bf = cv2.BFMatcher(cv2.NORM_HAMMING, crossCheck=True)
        matches = bf.match(des1, des2)
        matches = sorted(matches, key=lambda x: x.distance)

        if len(matches) < 20:
            print("Warning: Too few matching points, returning original image")
            return img2

        good_matches = matches[: min(len(matches), 100)]
        src_pts = np.float32([kp1[m.queryIdx].pt for m in good_matches]).reshape(
            -1, 1, 2
        )
        dst_pts = np.float32([kp2[m.trainIdx].pt for m in good_matches]).reshape(
            -1, 1, 2
        )

        try:
            M, mask = cv2.findHomography(
                dst_pts, src_pts, cv2.RANSAC, 3.0, maxIters=10000, confidence=0.99
            )

            if M is not None:
                height, width = img1.shape[:2]
                aligned_img2 = cv2.warpPerspective(img2, M, (width, height))
                return aligned_img2
            else:
                print(
                    "Warning: Unable to calculate homography matrix, returning original image"
                )
                return img2

        except Exception as e:
            print(f"Error during registration: {e}")
            return img2

    def filter_building_changes_enhanced(self, change_mask, min_area=3000):
        """
        Enhanced filtering focused on building-like changes
        """
        # Find connected components
        num_labels, labels, stats, centroids = cv2.connectedComponentsWithStats(
            change_mask
        )
        filtered_mask = np.zeros_like(change_mask)
        valid_changes = 0

        for i in range(1, num_labels):
            area = stats[i, cv2.CC_STAT_AREA]

            if area >= min_area:
                component_mask = (labels == i).astype(np.uint8) * 255
                contours, _ = cv2.findContours(
                    component_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
                )

                for contour in contours:
                    if self.is_building_like_shape(contour):
                        filtered_mask[labels == i] = 255
                        valid_changes += 1
                        break

        print(
            f"Found {valid_changes} significant building changes after enhanced filtering"
        )
        return filtered_mask

    def process_images(self, image1_path, image2_path, output_path=None):
        """
        Enhanced processing pipeline with improved shadow and vegetation removal
        """
        # Read images
        print(f"Reading images: {image1_path}, {image2_path}")
        img1 = cv2.imread(image1_path)
        img2 = cv2.imread(image2_path)

        if img1 is None or img2 is None:
            raise ValueError("Unable to read image files")

        # Image registration
        print("Performing image registration...")
        aligned_img2 = self.register_images(img1, img2)

        # Enhanced preprocessing with shadow correction
        print("Preprocessing images with shadow correction...")
        processed_img1 = self.preprocess_image(img1)
        processed_img2 = self.preprocess_image(aligned_img2)

        # Remove shadows first
        print("Detecting and removing shadows...")
        img1_no_shadow, shadow_mask1 = self.detect_and_remove_shadows(processed_img1)
        img2_no_shadow, shadow_mask2 = self.detect_and_remove_shadows(processed_img2)

        # Remove roads
        print("Removing roads...")
        img1_no_road, road_mask1 = self.remove_roads_comprehensive(img1_no_shadow)
        img2_no_road, road_mask2 = self.remove_roads_comprehensive(img2_no_shadow)

        # Enhanced vegetation removal
        print("Removing vegetation with enhanced detection...")
        img1_clean, veg_mask1 = self.remove_vegetation_enhanced(img1_no_road)
        img2_clean, veg_mask2 = self.remove_vegetation_enhanced(img2_no_road)

        # Enhanced building extraction
        print("Extracting building regions with shape analysis...")
        building_mask1 = self.extract_building_regions_enhanced(img1_clean)
        building_mask2 = self.extract_building_regions_enhanced(img2_clean)

        # Enhanced building change detection
        print("Detecting building changes with enhanced analysis...")
        change_mask = self.detect_building_changes_enhanced(
            img1_clean, img2_clean, building_mask1, building_mask2
        )

        # Apply morphological operations for building-like shapes
        print("Applying building-specific morphological operations...")
        building_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (8, 8))
        change_mask = cv2.morphologyEx(
            change_mask,
            cv2.MORPH_OPEN,
            cv2.getStructuringElement(cv2.MORPH_RECT, (5, 5)),
        )
        change_mask = cv2.morphologyEx(change_mask, cv2.MORPH_CLOSE, building_kernel)

        # Enhanced filtering for building changes only
        print("Filtering for building-like changes with shape analysis...")
        final_change_mask = self.filter_building_changes_enhanced(
            change_mask, min_area=3000
        )

        # Create result visualization
        result_image = self.create_result_image(aligned_img2, final_change_mask)

        # Display results
        self.visualize_results(result_image, output_path)

        # Enhanced statistics
        total_pixels = final_change_mask.shape[0] * final_change_mask.shape[1]
        changed_pixels = np.sum(final_change_mask > 0)
        change_percentage = (changed_pixels / total_pixels) * 100

        print(f"\n=== ENHANCED BUILDING CHANGE DETECTION RESULTS ===")
        print(f"Total building change pixels: {changed_pixels}")
        print(f"Building change percentage: {change_percentage:.3f}%")

        # Detailed analysis of detected changes
        num_labels, labels = cv2.connectedComponents(final_change_mask)
        print(f"Number of significant building changes detected: {num_labels-1}")

        # Analyze each detected change
        for i in range(1, num_labels):
            component_mask = (labels == i).astype(np.uint8) * 255
            contours, _ = cv2.findContours(
                component_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
            )

            for contour in contours:
                area = cv2.contourArea(contour)
                x, y, w, h = cv2.boundingRect(contour)
                aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

                print(
                    f"  Change {i}: Area={area:.0f}px, Size={w}x{h}px, "
                    f"Aspect Ratio={aspect_ratio:.2f}"
                )

        return final_change_mask, result_image

    def create_result_image(self, img2, change_mask):
        """
        Create result image with enhanced building change annotations
        """
        result_image = img2.copy()

        # Find contours of change areas
        contours, _ = cv2.findContours(
            change_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )

        # Annotate each building change with enhanced visualization
        for i, contour in enumerate(contours):
            # Draw thick red outline
            cv2.drawContours(result_image, [contour], -1, (0, 0, 255), 4)

            # Add semi-transparent overlay for better visibility
            overlay = result_image.copy()
            cv2.fillPoly(overlay, [contour], (0, 100, 255))  # Orange fill
            result_image = cv2.addWeighted(result_image, 0.8, overlay, 0.2, 0)

            # Calculate bounding rectangle and shape metrics
            x, y, w, h = cv2.boundingRect(contour)
            area = cv2.contourArea(contour)

            # Create detailed label
            aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0
            label = f"Building Change {i+1}"
            details = f"Area: {area:.0f}px, {w}x{h}px, AR: {aspect_ratio:.1f}"

            # Draw label background
            font_scale = 0.8
            thickness = 2

            # Main label
            label_size = cv2.getTextSize(
                label, cv2.FONT_HERSHEY_SIMPLEX, font_scale, thickness
            )[0]
            # Details label
            details_size = cv2.getTextSize(details, cv2.FONT_HERSHEY_SIMPLEX, 0.6, 1)[0]

            # Background rectangle
            max_width = max(label_size[0], details_size[0]) + 20
            total_height = label_size[1] + details_size[1] + 30

            cv2.rectangle(
                result_image,
                (x, y - total_height - 10),
                (x + max_width, y),
                (0, 0, 0),  # Black background
                -1,
            )

            # Draw border
            cv2.rectangle(
                result_image,
                (x, y - total_height - 10),
                (x + max_width, y),
                (0, 0, 255),  # Red border
                2,
            )

            # Draw main label text
            cv2.putText(
                result_image,
                label,
                (x + 10, y - details_size[1] - 15),
                cv2.FONT_HERSHEY_SIMPLEX,
                font_scale,
                (255, 255, 255),  # White text
                thickness,
            )

            # Draw details text
            cv2.putText(
                result_image,
                details,
                (x + 10, y - 5),
                cv2.FONT_HERSHEY_SIMPLEX,
                0.6,
                (200, 200, 200),  # Light gray text
                1,
            )

            # Add center point marker
            center_x = x + w // 2
            center_y = y + h // 2
            cv2.circle(
                result_image, (center_x, center_y), 5, (0, 255, 0), -1
            )  # Green dot
            cv2.circle(
                result_image, (center_x, center_y), 8, (255, 255, 255), 2
            )  # White circle

        return result_image

    def visualize_results(self, result_image, save_path=None):
        """
        Display the final result with enhanced visualization
        """
        plt.figure(figsize=(16, 12))
        plt.imshow(cv2.cvtColor(result_image, cv2.COLOR_BGR2RGB))
        plt.axis("off")

        # Add legend
        legend_elements = [
            plt.Rectangle(
                (0, 0), 1, 1, facecolor="red", alpha=0.7, label="Building Changes"
            ),
            plt.Circle((0, 0), 1, facecolor="green", label="Change Centers"),
            plt.Rectangle(
                (0, 0), 1, 1, facecolor="orange", alpha=0.3, label="Change Areas"
            ),
        ]
        plt.legend(handles=legend_elements, loc="upper right", fontsize=12)

        if save_path:
            plt.savefig(save_path, dpi=300, bbox_inches="tight")
            print(f"Enhanced result saved to: {save_path}")

        plt.show()

        return result_image

    def create_diagnostic_visualization(
        self,
        img1,
        img2,
        shadow_mask1,
        shadow_mask2,
        veg_mask1,
        veg_mask2,
        building_mask1,
        building_mask2,
        final_changes,
    ):
        """
        Create diagnostic visualization showing intermediate results
        """
        fig, axes = plt.subplots(3, 4, figsize=(20, 15))

        # Row 1: Original images and shadows
        axes[0, 0].imshow(cv2.cvtColor(img1, cv2.COLOR_BGR2RGB))
        axes[0, 0].set_title("Original Image 1")
        axes[0, 0].axis("off")

        axes[0, 1].imshow(cv2.cvtColor(img2, cv2.COLOR_BGR2RGB))
        axes[0, 1].set_title("Original Image 2")
        axes[0, 1].axis("off")

        axes[0, 2].imshow(shadow_mask1, cmap="gray")
        axes[0, 2].set_title("Shadow Mask 1")
        axes[0, 2].axis("off")

        axes[0, 3].imshow(shadow_mask2, cmap="gray")
        axes[0, 3].set_title("Shadow Mask 2")
        axes[0, 3].axis("off")

        # Row 2: Vegetation masks and building masks
        axes[1, 0].imshow(veg_mask1, cmap="Greens")
        axes[1, 0].set_title("Vegetation Mask 1")
        axes[1, 0].axis("off")

        axes[1, 1].imshow(veg_mask2, cmap="Greens")
        axes[1, 1].set_title("Vegetation Mask 2")
        axes[1, 1].axis("off")

        axes[1, 2].imshow(building_mask1, cmap="Blues")
        axes[1, 2].set_title("Building Mask 1")
        axes[1, 2].axis("off")

        axes[1, 3].imshow(building_mask2, cmap="Blues")
        axes[1, 3].set_title("Building Mask 2")
        axes[1, 3].axis("off")

        # Row 3: Final results
        axes[2, 0].imshow(final_changes, cmap="Reds")
        axes[2, 0].set_title("Final Building Changes")
        axes[2, 0].axis("off")

        # Overlay changes on original
        overlay1 = img1.copy()
        overlay1[final_changes > 0] = [0, 0, 255]
        axes[2, 1].imshow(cv2.cvtColor(overlay1, cv2.COLOR_BGR2RGB))
        axes[2, 1].set_title("Changes on Image 1")
        axes[2, 1].axis("off")

        overlay2 = img2.copy()
        overlay2[final_changes > 0] = [0, 0, 255]
        axes[2, 2].imshow(cv2.cvtColor(overlay2, cv2.COLOR_BGR2RGB))
        axes[2, 2].set_title("Changes on Image 2")
        axes[2, 2].axis("off")

        # Statistics plot
        axes[2, 3].axis("off")
        stats_text = f"""Detection Statistics:

Shadow Pixels 1: {np.sum(shadow_mask1 > 0):,}
Shadow Pixels 2: {np.sum(shadow_mask2 > 0):,}

Vegetation Pixels 1: {np.sum(veg_mask1 > 0):,}
Vegetation Pixels 2: {np.sum(veg_mask2 > 0):,}

Building Pixels 1: {np.sum(building_mask1 > 0):,}
Building Pixels 2: {np.sum(building_mask2 > 0):,}

Final Changes: {np.sum(final_changes > 0):,}
Change Percentage: {(np.sum(final_changes > 0) / (final_changes.shape[0] * final_changes.shape[1]) * 100):.3f}%
        """
        axes[2, 3].text(
            0.1,
            0.9,
            stats_text,
            fontsize=10,
            verticalalignment="top",
            fontfamily="monospace",
        )

        plt.tight_layout()
        plt.savefig("diagnostic_results.png", dpi=200, bbox_inches="tight")
        plt.show()


# Enhanced usage example
def main():
    """
    Main function for enhanced building-focused change detection
    """
    detector = EnhancedBuildingChangeDetector()

    # Process images (replace with actual image paths)
    image1_path = "./images/image_1.jpg"  # Reference image
    image2_path = "./images/image_2.jpg"  # Current image
    output_path = "./images/enhanced_building_changes_result.png"  # Result save path

    try:
        change_mask, result_image = detector.process_images(
            image1_path, image2_path, output_path
        )

        # Save results with enhanced naming
        if output_path:
            cv2.imwrite("enhanced_building_change_mask.png", change_mask)
            cv2.imwrite("enhanced_building_changes_annotated.png", result_image)
            print(f"\nEnhanced results saved:")
            print(f"- Main result: {output_path}")
            print(f"- Change mask: enhanced_building_change_mask.png")
            print(f"- Annotated result: enhanced_building_changes_annotated.png")
            print(f"- Diagnostic visualization: diagnostic_results.png")

    except Exception as e:
        print(f"Error during processing: {e}")
        import traceback

        traceback.print_exc()


if __name__ == "__main__":
    main()
