import cv2
import numpy as np
from sklearn.cluster import KMeans
import matplotlib.pyplot as plt
from skimage import segmentation, measure
import os


class SuperEnhancedBuildingChangeDetector:
    def __init__(self):
        """
        Super Enhanced Building-Focused Change Detection System
        Advanced tree filtering and building shape analysis
        """
        self.kernel_size = 7
        self.morph_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (7, 7))

    def preprocess_image(self, image):
        """
        Enhanced image preprocessing with shadow correction
        """
        # Gaussian filtering for noise reduction
        denoised = cv2.GaussianBlur(image, (5, 5), 0)

        # Shadow correction using adaptive histogram equalization
        if len(image.shape) == 3:
            # Convert to LAB color space for better shadow handling
            lab = cv2.cvtColor(denoised, cv2.COLOR_BGR2LAB)

            # Apply CLAHE to L channel to reduce shadow effects
            clahe = cv2.createCLAHE(clipLimit=3.0, tileGridSize=(8, 8))
            lab[:, :, 0] = clahe.apply(lab[:, :, 0])
            enhanced = cv2.cvtColor(lab, cv2.COLOR_LAB2BGR)
        else:
            clahe = cv2.createCLAHE(clipLimit=3.0, tileGridSize=(8, 8))
            enhanced = clahe.apply(denoised)

        return enhanced

    def detect_and_remove_shadows(self, image):
        """
        Advanced shadow detection and removal
        """
        # Convert to multiple color spaces for comprehensive shadow detection
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)
        yuv = cv2.cvtColor(image, cv2.COLOR_BGR2YUV)

        shadow_mask = np.zeros(image.shape[:2], dtype=np.uint8)

        # Method 1: Low brightness detection in HSV
        v_channel = hsv[:, :, 2]
        low_brightness = v_channel < 80
        shadow_mask1 = np.uint8(low_brightness * 255)

        # Method 2: Blue dominance (shadows often have blue cast)
        b, g, r = cv2.split(image)
        blue_dominant = (b > g) & (b > r) & (b > 60)
        shadow_mask2 = np.uint8(blue_dominant * 255)

        # Method 3: Low saturation + low value
        s_channel = hsv[:, :, 1]
        low_sat_val = (s_channel < 50) & (v_channel < 100)
        shadow_mask3 = np.uint8(low_sat_val * 255)

        # Method 4: Using LAB color space - shadows have specific characteristics
        a_channel = lab[:, :, 1]
        b_channel = lab[:, :, 2]
        l_channel = lab[:, :, 0]

        # Shadows typically have low L and shifted A/B values
        shadow_lab = (l_channel < 80) & (a_channel > 120) & (b_channel < 135)
        shadow_mask4 = np.uint8(shadow_lab * 255)

        # Combine all shadow detection methods
        shadow_mask = cv2.bitwise_or(shadow_mask1, shadow_mask2)
        shadow_mask = cv2.bitwise_or(shadow_mask, shadow_mask3)
        shadow_mask = cv2.bitwise_or(shadow_mask, shadow_mask4)

        # Morphological operations to clean up shadow mask
        shadow_kernel = cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (15, 15))
        shadow_mask = cv2.morphologyEx(shadow_mask, cv2.MORPH_CLOSE, shadow_kernel)
        shadow_mask = cv2.morphologyEx(
            shadow_mask,
            cv2.MORPH_OPEN,
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (8, 8)),
        )

        # Remove shadows from image
        result = image.copy()

        # Instead of setting to gray, try to brighten shadow areas
        shadow_indices = shadow_mask > 0
        if np.any(shadow_indices):
            # Brighten shadows by increasing brightness
            hsv_temp = cv2.cvtColor(result, cv2.COLOR_BGR2HSV)
            hsv_temp[shadow_indices, 2] = np.minimum(
                255, hsv_temp[shadow_indices, 2] * 1.5
            )
            result = cv2.cvtColor(hsv_temp, cv2.COLOR_HSV2BGR)

        return result, shadow_mask

    def remove_vegetation_super_enhanced(self, image):
        """
        SUPER ENHANCED vegetation removal with aggressive tree detection
        """
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)
        gray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)

        vegetation_mask = np.zeros(hsv.shape[:2], dtype=np.uint8)

        # Method 1: EXPANDED green detection ranges (more aggressive)
        green_ranges = [
            ([15, 15, 10], [105, 255, 250]),  # VERY broad green (catch all greens)
            ([20, 20, 15], [100, 255, 240]),  # Extended range
            ([25, 25, 20], [95, 255, 230]),   # Standard green
            ([30, 30, 25], [85, 255, 220]),   # Medium green
            ([35, 35, 30], [80, 255, 200]),   # Dark green (tree shadows)
            ([40, 40, 35], [75, 255, 180]),   # Specific tree green
            ([45, 50, 40], [70, 200, 160]),   # Deep forest green
            ([50, 60, 45], [65, 180, 140]),   # Very dark tree green
        ]

        for lower, upper in green_ranges:
            mask = cv2.inRange(hsv, np.array(lower), np.array(upper))
            vegetation_mask = cv2.bitwise_or(vegetation_mask, mask)

        # Method 2: Enhanced LAB color space detection
        a_channel = lab[:, :, 1]
        b_channel = lab[:, :, 2]
        l_channel = lab[:, :, 0]

        # More aggressive green detection in LAB
        green_lab_conditions = [
            (a_channel < 128) & (b_channel > 115) & (l_channel > 25),  # Standard
            (a_channel < 125) & (b_channel > 118) & (l_channel > 20),  # Darker greens
            (a_channel < 130) & (b_channel > 112) & (l_channel > 30),  # Brighter greens
            (a_channel < 122) & (b_channel > 120) & (l_channel > 15),  # Very dark greens
        ]

        for condition in green_lab_conditions:
            green_lab_mask = np.uint8(condition * 255)
            vegetation_mask = cv2.bitwise_or(vegetation_mask, green_lab_mask)

        # Method 3: Enhanced NDVI-like index with multiple thresholds
        b_bgr, g_bgr, r_bgr = cv2.split(image)

        # Multiple NDVI-like calculations
        denominator1 = r_bgr.astype(np.float32) + g_bgr.astype(np.float32)
        denominator1[denominator1 == 0] = 1
        ndvi_like1 = (g_bgr.astype(np.float32) - r_bgr.astype(np.float32)) / denominator1

        # Alternative NDVI calculation
        denominator2 = (r_bgr.astype(np.float32) + g_bgr.astype(np.float32) + b_bgr.astype(np.float32)) / 3
        denominator2[denominator2 == 0] = 1
        ndvi_like2 = (g_bgr.astype(np.float32) - denominator2) / (g_bgr.astype(np.float32) + denominator2)

        # Combine NDVI calculations with multiple thresholds
        ndvi_conditions = [
            (ndvi_like1 > 0.05) & (g_bgr > 40),   # Lower threshold
            (ndvi_like1 > 0.1) & (g_bgr > 35),    # Standard threshold
            (ndvi_like1 > 0.15) & (g_bgr > 30),   # Higher threshold
            (ndvi_like2 > 0.02) & (g_bgr > 45),   # Alternative calculation
        ]

        for condition in ndvi_conditions:
            ndvi_mask = np.uint8(condition * 255)
            vegetation_mask = cv2.bitwise_or(vegetation_mask, ndvi_mask)

        # Method 4: SUPER AGGRESSIVE texture-based tree detection
        # Trees have very specific organic, irregular textures

        # Multiple Gabor filters for organic patterns
        def apply_gabor_filter(img, theta, frequency=0.1):
            kernel = cv2.getGaborKernel(
                (31, 31), 8, theta, 2*np.pi*frequency, 0.5, 0, ktype=cv2.CV_32F
            )
            return cv2.filter2D(img, cv2.CV_8UC3, kernel)

        # Apply Gabor filters at multiple frequencies and orientations
        gabor_responses = []
        frequencies = [0.05, 0.1, 0.15, 0.2]  # Multiple frequencies for different tree textures

        for freq in frequencies:
            for angle in [0, 30, 45, 60, 90, 120, 135, 150]:
                gabor_resp = apply_gabor_filter(gray, np.radians(angle), freq)
                gabor_responses.append(gabor_resp)

        # Combine all Gabor responses
        gabor_combined = np.zeros_like(gray, dtype=np.float32)
        for resp in gabor_responses:
            gabor_combined += np.abs(resp.astype(np.float32))

        # Normalize and threshold
        gabor_combined = (gabor_combined / len(gabor_responses)).astype(np.uint8)
        _, organic_texture = cv2.threshold(gabor_combined, 25, 255, cv2.THRESH_BINARY)

        # Method 5: Enhanced edge density analysis (trees have very high edge density)
        edges = cv2.Canny(gray, 30, 100)  # Lower thresholds for more edges

        # Multiple kernel sizes for edge density
        edge_kernels = [
            cv2.getStructuringElement(cv2.MORPH_RECT, (10, 10)),
            cv2.getStructuringElement(cv2.MORPH_RECT, (15, 15)),
            cv2.getStructuringElement(cv2.MORPH_RECT, (20, 20)),
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (12, 12)),
        ]

        edge_density_masks = []
        for kernel in edge_kernels:
            edge_density = cv2.morphologyEx(edges, cv2.MORPH_CLOSE, kernel)
            edge_density_masks.append(edge_density)

        # Combine edge density masks
        combined_edge_density = np.zeros_like(gray)
        for mask in edge_density_masks:
            combined_edge_density = cv2.bitwise_or(combined_edge_density, mask)

        # Method 6: Tree-specific color patterns
        # Trees often have brown trunks and green leaves - detect this combination
        brown_ranges = [
            ([10, 50, 20], [25, 200, 150]),   # Brown trunks
            ([8, 40, 30], [28, 180, 140]),    # Dark brown
        ]

        brown_mask = np.zeros_like(gray)
        for lower, upper in brown_ranges:
            mask = cv2.inRange(hsv, np.array(lower), np.array(upper))
            brown_mask = cv2.bitwise_or(brown_mask, mask)

        # Dilate brown areas to include nearby green (tree crowns)
        brown_dilated = cv2.dilate(brown_mask,
                                  cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (40, 40)),
                                  iterations=2)

        # Method 7: Height/elevation simulation using brightness gradients
        # Trees often show brightness variations due to their 3D structure
        sobelx = cv2.Sobel(gray, cv2.CV_64F, 1, 0, ksize=3)
        sobely = cv2.Sobel(gray, cv2.CV_64F, 0, 1, ksize=3)
        gradient_magnitude = np.sqrt(sobelx**2 + sobely**2)
        gradient_magnitude = (gradient_magnitude * 255 / gradient_magnitude.max()).astype(np.uint8)

        # High gradient areas combined with green indicate trees
        _, high_gradient = cv2.threshold(gradient_magnitude, 40, 255, cv2.THRESH_BINARY)

        # COMBINE ALL VEGETATION DETECTION METHODS
        vegetation_mask = cv2.bitwise_or(vegetation_mask, organic_texture)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, combined_edge_density)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, brown_dilated)

        # Only include high gradient areas that are also green
        green_high_gradient = cv2.bitwise_and(vegetation_mask, high_gradient)
        vegetation_mask = cv2.bitwise_or(vegetation_mask, green_high_gradient)

        # SUPER AGGRESSIVE morphological operations
        vegetation_kernels = [
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (25, 25)),  # Large closing
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (35, 35)),  # Very large closing
            cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (15, 15)),  # Medium opening
        ]

        # Apply multiple closing operations to connect tree parts
        for i, kernel in enumerate(vegetation_kernels[:2]):  # Use first 2 for closing
            vegetation_mask = cv2.morphologyEx(vegetation_mask, cv2.MORPH_CLOSE, kernel)

        # Apply opening to remove noise
        vegetation_mask = cv2.morphologyEx(vegetation_mask, cv2.MORPH_OPEN, vegetation_kernels[2])

        # Remove small noise but keep larger vegetation areas
        contours, _ = cv2.findContours(
            vegetation_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )
        cleaned_mask = np.zeros_like(vegetation_mask)

        for contour in contours:
            area = cv2.contourArea(contour)
            if area > 200:  # Even smaller threshold to catch more vegetation
                # Additional check: is this likely vegetation based on shape?
                if self.is_vegetation_like_shape(contour):
                    cv2.fillPoly(cleaned_mask, [contour], 255)

        # Apply vegetation mask
        result = image.copy()
        result[cleaned_mask > 0] = [128, 128, 128]

        return result, cleaned_mask

    def is_vegetation_like_shape(self, contour):
        """
        Determine if a contour represents vegetation (trees, bushes)
        Trees have irregular, organic shapes unlike buildings
        """
        area = cv2.contourArea(contour)
        if area < 200:
            return False

        # Get shape characteristics
        hull = cv2.convexHull(contour)
        hull_area = cv2.contourArea(hull)

        # Calculate metrics
        solidity = area / hull_area if hull_area > 0 else 0
        perimeter = cv2.arcLength(contour, True)

        # Trees are typically:
        # - Less solid (more irregular edges) than buildings
        # - Have more complex perimeters
        # - Are not perfectly rectangular

        x, y, w, h = cv2.boundingRect(contour)
        rect_area = w * h
        extent = area / rect_area if rect_area > 0 else 0
        aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

        # Trees have:
        # - Lower solidity (more irregular) - solidity < 0.85
        # - More complex shape - extent < 0.8
        # - Can be more elongated than buildings
        # - Higher perimeter to area ratio
        circularity = 4 * np.pi * area / (perimeter * perimeter) if perimeter > 0 else 0

        is_vegetation = (
            solidity < 0.85 and          # Irregular shape
            extent < 0.8 and             # Not rectangular
            circularity < 0.6            # Not circular/regular
        )

        return is_vegetation

    def extract_building_regions_enhanced(self, image):
        """
        Enhanced building extraction with STRONGER anti-tree filtering
        """
        gray = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)

        # Method 1: Enhanced line detection for building edges
        edges = cv2.Canny(gray, 30, 150, apertureSize=3)

        # Focus on STRAIGHT lines (buildings) vs organic curves (trees)
        lines_long = cv2.HoughLinesP(
            edges, 1, np.pi / 180, threshold=100, minLineLength=80, maxLineGap=10
        )
        lines_short = cv2.HoughLinesP(
            edges, 1, np.pi / 180, threshold=50, minLineLength=40, maxLineGap=8
        )

        line_mask = np.zeros_like(gray)

        # Only keep STRAIGHT lines (filter out curved tree edges)
        def is_straight_line(x1, y1, x2, y2, tolerance=5):
            """Check if line segment is sufficiently straight"""
            length = np.sqrt((x2-x1)**2 + (y2-y1)**2)
            return length > 20  # Minimum length for building lines

        # Draw long straight lines (building edges)
        if lines_long is not None:
            for line in lines_long:
                x1, y1, x2, y2 = line[0]
                if is_straight_line(x1, y1, x2, y2):
                    cv2.line(line_mask, (x1, y1), (x2, y2), 255, 3)

        # Draw short straight lines (building details)
        if lines_short is not None:
            for line in lines_short:
                x1, y1, x2, y2 = line[0]
                if is_straight_line(x1, y1, x2, y2):
                    cv2.line(line_mask, (x1, y1), (x2, y2), 128, 2)

        # Method 2: ENHANCED corner detection - buildings have sharp corners, trees don't
        corners_fine = cv2.cornerHarris(gray, 2, 3, 0.04)
        corners_coarse = cv2.cornerHarris(gray, 5, 7, 0.04)

        corner_mask = np.zeros_like(gray)
        # Use higher threshold to only get sharp corners (buildings)
        corner_mask[corners_fine > 0.02 * corners_fine.max()] = 255  # Increased threshold
        corner_mask[corners_coarse > 0.02 * corners_coarse.max()] = 255

        # Method 3: STRICT rectangular pattern detection
        rect_kernels = [
            cv2.getStructuringElement(cv2.MORPH_RECT, (40, 10)),  # Large horizontal
            cv2.getStructuringElement(cv2.MORPH_RECT, (10, 40)),  # Large vertical
            cv2.getStructuringElement(cv2.MORPH_RECT, (30, 8)),   # Medium horizontal
            cv2.getStructuringElement(cv2.MORPH_RECT, (8, 30)),   # Medium vertical
        ]

        rectangular_features = np.zeros_like(gray)
        for kernel in rect_kernels:
            rect_resp = cv2.morphologyEx(edges, cv2.MORPH_CLOSE, kernel)
            rectangular_features = cv2.bitwise_or(rectangular_features, rect_resp)

        # Method 4: ENHANCED template matching for ONLY building shapes
        templates = []

        # More specific building templates
        # Large square template
        square_template = np.zeros((60, 60), dtype=np.uint8)
        cv2.rectangle(square_template, (5, 5), (55, 55), 255, 3)
        templates.append(square_template)

        # Large rectangle template
        rect_template1 = np.zeros((40, 80), dtype=np.uint8)
        cv2.rectangle(rect_template1, (5, 5), (75, 35), 255, 3)
        templates.append(rect_template1)

        # Vertical rectangle template
        rect_template2 = np.zeros((80, 40), dtype=np.uint8)
        cv2.rectangle(rect_template2, (5, 5), (35, 75), 255, 3)
        templates.append(rect_template2)

        template_mask = np.zeros_like(gray)
        for template in templates:
            result = cv2.matchTemplate(gray, template, cv2.TM_CCOEFF_NORMED)
            locations = np.where(result >= 0.4)  # Higher threshold for better matches
            for pt in zip(*locations[::-1]):
                cv2.rectangle(
                    template_mask,
                    pt,
                    (pt[0] + template.shape[1], pt[1] + template.shape[0]),
                    255,
                    -1,
                )

        # Method 5: Combine building features with STRICT filtering
        building_features = cv2.bitwise_or(line_mask, corner_mask)
        building_features = cv2.bitwise_or(building_features, rectangular_features)
        building_features = cv2.bitwise_or(building_features, template_mask)

        # STRICT morphological operations - only keep building-like shapes
        building_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (15, 15))  # Smaller kernel
        building_mask = cv2.morphologyEx(
            building_features, cv2.MORPH_CLOSE, building_kernel
        )

        # VERY STRICT filtering - only keep shapes that are definitely buildings
        contours, _ = cv2.findContours(
            building_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )
        filtered_mask = np.zeros_like(building_mask)

        for contour in contours:
            area = cv2.contourArea(contour)
            if area > 3000:  # Larger minimum area
                # STRICT building shape check
                if self.is_definitely_building_shape(contour):
                    cv2.fillPoly(filtered_mask, [contour], 255)

        return filtered_mask

    def is_definitely_building_shape(self, contour):
        """
        VERY STRICT building shape analysis - must clearly be a building, not a tree
        """
        area = cv2.contourArea(contour)
        if area < 3000:  # Increased minimum area
            return False

        # Get bounding rectangle and shape metrics
        x, y, w, h = cv2.boundingRect(contour)
        rect_area = w * h

        # Calculate shape metrics
        extent = area / rect_area if rect_area > 0 else 0
        aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

        # Calculate hull ratio (convexity)
        hull = cv2.convexHull(contour)
        hull_area = cv2.contourArea(hull)
        solidity = area / hull_area if hull_area > 0 else 0

        # Calculate perimeter metrics
        perimeter = cv2.arcLength(contour, True)
        circularity = 4 * np.pi * area / (perimeter * perimeter) if perimeter > 0 else 0

        # VERY STRICT criteria for buildings:
        # - Must be quite rectangular (extent > 0.6)
        # - Cannot be too elongated (aspect_ratio < 4)
        # - Must be very convex (solidity > 0.8)
        # - Must have reasonable circularity (not too complex)
        # - Must be large enough to be a significant building

        is_building = (
            extent > 0.6 and           # Very rectangular
            aspect_ratio < 4.0 and     # Not too elongated
            solidity > 0.8 and         # Very convex (not organic)
            circularity > 0.15 and     # Not too complex
            area > 3000 and            # Large enough
            min(w, h) > 30             # Minimum dimension
        )

        return is_building

    def detect_building_changes_enhanced(
        self, img1, img2, building_mask1, building_mask2
    ):
        """
        Enhanced building change detection with tree change filtering
        """
        # 1. AREA CHANGES
        area_diff = cv2.absdiff(building_mask1, building_mask2)

        # 2. COLOR CHANGES with building-specific analysis
        hsv1 = cv2.cvtColor(img1, cv2.COLOR_BGR2HSV)
        hsv2 = cv2.cvtColor(img2, cv2.COLOR_BGR2HSV)
        lab1 = cv2.cvtColor(img1, cv2.COLOR_BGR2LAB)
        lab2 = cv2.cvtColor(img2, cv2.COLOR_BGR2LAB)

        # Enhanced color difference detection
        hsv_diff = cv2.absdiff(hsv1, hsv2)
        hsv_change_gray = cv2.cvtColor(hsv_diff, cv2.COLOR_HSV2BGR)
        hsv_change_gray = cv2.cvtColor(hsv_change_gray, cv2.COLOR_BGR2GRAY)

        lab_diff = cv2.absdiff(lab1, lab2)
        lab_change_gray = cv2.cvtColor(lab_diff, cv2.COLOR_LAB2BGR)
        lab_change_gray = cv2.cvtColor(lab_change_gray, cv2.COLOR_BGR2GRAY)

        # Hue-specific changes
        h1, h2 = hsv1[:, :, 0], hsv2[:, :, 0]
        hue_diff = np.abs(h1.astype(np.float32) - h2.astype(np.float32))
        hue_diff = np.minimum(hue_diff, 180 - hue_diff)
        hue_diff = (hue_diff * 255 / 90).astype(np.uint8)

        # Combine color changes
        color_change = cv2.addWeighted(hsv_change_gray, 0.3, lab_change_gray, 0.5, 0)
        color_change = cv2.addWeighted(color_change, 0.8, hue_diff, 0.2, 0)

        # Only consider changes within building areas
        combined_building_mask = cv2.bitwise_or(building_mask1, building_mask2)
        building_color_change = cv2.bitwise_and(color_change, combined_building_mask)

        # 3. TEXTURE CHANGES
        gray1 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
        gray2 = cv2.cvtColor(img2, cv2.COLOR_BGR2GRAY)

        def get_texture_features(img):
            laplacian = cv2.Laplacian(img, cv2.CV_64F)
            return np.absolute(laplacian).astype(np.uint8)

        texture1 = get_texture_features(gray1)
        texture2 = get_texture_features(gray2)
        texture_diff = cv2.absdiff(texture1, texture2)

        building_texture_change = cv2.bitwise_and(texture_diff, combined_building_mask)

        # 4. COMBINE ALL CHANGES
        final_change = cv2.addWeighted(area_diff, 0.5, building_color_change, 0.3, 0)
        final_change = cv2.addWeighted(
            final_change, 0.8, building_texture_change, 0.2, 0
        )

        # Apply higher threshold for more significant changes only
        _, change_mask = cv2.threshold(final_change, 50, 255, cv2.THRESH_BINARY)  # Increased threshold

        return change_mask

    def remove_roads_comprehensive(self, image):
        """
        Comprehensive road removal
        """
        hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)
        lab = cv2.cvtColor(image, cv2.COLOR_BGR2LAB)

        road_mask = np.zeros(image.shape[:2], dtype=np.uint8)

        # Gray roads detection
        gray_ranges_bgr = [
            ([60, 60, 60], [140, 140, 140]),
            ([40, 40, 40], [100, 100, 100]),
            ([100, 100, 100], [180, 180, 180]),
            ([20, 20, 20], [70, 70, 70]),
        ]

        for lower, upper in gray_ranges_bgr:
            mask = cv2.inRange(image, np.array(lower), np.array(upper))
            road_mask = cv2.bitwise_or(road_mask, mask)

        # Low saturation areas
        low_saturation = hsv[:, :, 1] < 40
        low_sat_mask = np.uint8(low_saturation * 255)

        # Yellow line detection
        yellow_lower = np.array([20, 100, 100])
        yellow_upper = np.array([30, 255, 255])
        yellow_mask = cv2.inRange(hsv, yellow_lower, yellow_upper)
        yellow_dilated = cv2.dilate(
            yellow_mask,
            cv2.getStructuringElement(cv2.MORPH_RECT, (20, 20)),
            iterations=2,
        )

        # Combine road detection methods
        combined_road_mask = cv2.bitwise_or(road_mask, low_sat_mask)
        combined_road_mask = cv2.bitwise_or(combined_road_mask, yellow_dilated)

        # Linear structure detection
        kernels = [
            cv2.getStructuringElement(cv2.MORPH_RECT, (40, 5)),
            cv2.getStructuringElement(cv2.MORPH_RECT, (5, 40)),
        ]

        linear_structures = np.zeros_like(combined_road_mask)
        for kernel in kernels:
            linear_result = cv2.morphologyEx(combined_road_mask, cv2.MORPH_OPEN, kernel)
            linear_structures = cv2.bitwise_or(linear_structures, linear_result)

        final_road_mask = cv2.bitwise_or(combined_road_mask, linear_structures)

        # Clean up road mask
        kernel_close = cv2.getStructuringElement(cv2.MORPH_RECT, (15, 15))
        final_road_mask = cv2.morphologyEx(
            final_road_mask, cv2.MORPH_CLOSE, kernel_close
        )

        # Apply road mask
        result = image.copy()
        result[final_road_mask > 0] = [128, 128, 128]

        return result, final_road_mask

    def register_images(self, img1, img2):
        """
        Image registration using ORB features
        """
        gray1 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
        gray2 = cv2.cvtColor(img2, cv2.COLOR_BGR2GRAY)

        orb = cv2.ORB_create(nfeatures=2000)
        kp1, des1 = orb.detectAndCompute(gray1, None)
        kp2, des2 = orb.detectAndCompute(gray2, None)

        if des1 is None or des2 is None:
            print("Warning: Unable to extract sufficient feature points, returning original image")
            return img2

        bf = cv2.BFMatcher(cv2.NORM_HAMMING, crossCheck=True)
        matches = bf.match(des1, des2)
        matches = sorted(matches, key=lambda x: x.distance)

        if len(matches) < 20:
            print("Warning: Too few matching points, returning original image")
            return img2

        good_matches = matches[: min(len(matches), 100)]
        src_pts = np.float32([kp1[m.queryIdx].pt for m in good_matches]).reshape(-1, 1, 2)
        dst_pts = np.float32([kp2[m.trainIdx].pt for m in good_matches]).reshape(-1, 1, 2)

        try:
            M, mask = cv2.findHomography(
                dst_pts, src_pts, cv2.RANSAC, 3.0, maxIters=10000, confidence=0.99
            )

            if M is not None:
                height, width = img1.shape[:2]
                aligned_img2 = cv2.warpPerspective(img2, M, (width, height))
                return aligned_img2
            else:
                print("Warning: Unable to calculate homography matrix, returning original image")
                return img2

        except Exception as e:
            print(f"Error during registration: {e}")
            return img2

    def filter_tree_changes_aggressively(self, change_mask, img1, img2):
        """
        SUPER AGGRESSIVE tree change filtering
        This method specifically targets and removes tree-like changes
        """
        # Find all change regions
        contours, _ = cv2.findContours(change_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)

        filtered_mask = np.zeros_like(change_mask)
        tree_changes_removed = 0

        for contour in contours:
            area = cv2.contourArea(contour)
            if area < 1000:  # Skip very small changes
                continue

            # Create mask for this specific change region
            single_change_mask = np.zeros_like(change_mask)
            cv2.fillPoly(single_change_mask, [contour], 255)

            # Extract the region from both images
            x, y, w, h = cv2.boundingRect(contour)

            # Add padding around the region
            pad = 20
            x_start = max(0, x - pad)
            y_start = max(0, y - pad)
            x_end = min(img1.shape[1], x + w + pad)
            y_end = min(img1.shape[0], y + h + pad)

            region1 = img1[y_start:y_end, x_start:x_end]
            region2 = img2[y_start:y_end, x_start:x_end]
            region_mask = single_change_mask[y_start:y_end, x_start:x_end]

            # Analyze if this change is tree-like
            is_tree_change = self.analyze_if_tree_change(region1, region2, region_mask, contour)

            if not is_tree_change:
                # Keep this change - it's likely a building
                cv2.fillPoly(filtered_mask, [contour], 255)
            else:
                tree_changes_removed += 1
                print(f"Removed tree change {tree_changes_removed}: Area={area:.0f}px")

        print(f"Aggressively filtered out {tree_changes_removed} tree-like changes")
        return filtered_mask

    def analyze_if_tree_change(self, region1, region2, region_mask, contour):
        """
        Analyze if a change region represents tree changes vs building changes
        Returns True if it's likely a tree change (should be filtered out)
        """
        if region1.size == 0 or region2.size == 0:
            return True  # Remove invalid regions

        # 1. SHAPE ANALYSIS
        area = cv2.contourArea(contour)
        hull = cv2.convexHull(contour)
        hull_area = cv2.contourArea(hull)
        solidity = area / hull_area if hull_area > 0 else 0

        x, y, w, h = cv2.boundingRect(contour)
        rect_area = w * h
        extent = area / rect_area if rect_area > 0 else 0
        aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

        perimeter = cv2.arcLength(contour, True)
        circularity = 4 * np.pi * area / (perimeter * perimeter) if perimeter > 0 else 0

        # Trees have irregular shapes
        shape_score = 0
        if solidity < 0.7:  # Irregular shape
            shape_score += 2
        if extent < 0.6:   # Not rectangular
            shape_score += 2
        if circularity < 0.3:  # Complex perimeter
            shape_score += 1

        # 2. COLOR ANALYSIS
        # Extract colors from the change region
        mask_indices = region_mask > 0
        if not np.any(mask_indices):
            return True

        # Check for green colors in both regions
        hsv1 = cv2.cvtColor(region1, cv2.COLOR_BGR2HSV)
        hsv2 = cv2.cvtColor(region2, cv2.COLOR_BGR2HSV)

        green_pixels1 = self.count_green_pixels(hsv1[mask_indices])
        green_pixels2 = self.count_green_pixels(hsv2[mask_indices])
        total_pixels = np.sum(mask_indices)

        green_ratio1 = green_pixels1 / total_pixels if total_pixels > 0 else 0
        green_ratio2 = green_pixels2 / total_pixels if total_pixels > 0 else 0

        color_score = 0
        # High green content suggests trees
        if green_ratio1 > 0.3 or green_ratio2 > 0.3:
            color_score += 3
        if green_ratio1 > 0.5 or green_ratio2 > 0.5:
            color_score += 2

        # Check for brown colors (tree trunks/branches)
        brown_pixels1 = self.count_brown_pixels(hsv1[mask_indices])
        brown_pixels2 = self.count_brown_pixels(hsv2[mask_indices])

        brown_ratio1 = brown_pixels1 / total_pixels if total_pixels > 0 else 0
        brown_ratio2 = brown_pixels2 / total_pixels if total_pixels > 0 else 0

        if brown_ratio1 > 0.1 or brown_ratio2 > 0.1:
            color_score += 2

        # 3. TEXTURE ANALYSIS
        gray1 = cv2.cvtColor(region1, cv2.COLOR_BGR2GRAY)
        gray2 = cv2.cvtColor(region2, cv2.COLOR_BGR2GRAY)

        # Calculate texture complexity (trees have high texture complexity)
        def calculate_texture_complexity(gray_img):
            if gray_img.size == 0:
                return 0
            # Use Laplacian variance as texture measure
            laplacian = cv2.Laplacian(gray_img, cv2.CV_64F)
            return laplacian.var()

        texture1 = calculate_texture_complexity(gray1)
        texture2 = calculate_texture_complexity(gray2)

        texture_score = 0
        # High texture complexity suggests organic/tree content
        if texture1 > 1000 or texture2 > 1000:
            texture_score += 2
        if texture1 > 2000 or texture2 > 2000:
            texture_score += 1

        # 4. EDGE DENSITY ANALYSIS
        edges1 = cv2.Canny(gray1, 50, 150)
        edges2 = cv2.Canny(gray2, 50, 150)

        edge_density1 = np.sum(edges1 > 0) / edges1.size if edges1.size > 0 else 0
        edge_density2 = np.sum(edges2 > 0) / edges2.size if edges2.size > 0 else 0

        edge_score = 0
        # Trees typically have higher edge density than buildings
        if edge_density1 > 0.1 or edge_density2 > 0.1:
            edge_score += 2

        # 5. SIZE AND POSITION ANALYSIS
        size_score = 0
        # Very small changes are more likely to be tree movement
        if area < 2000:
            size_score += 1
        # Very elongated shapes might be tree branches
        if aspect_ratio > 5:
            size_score += 2

        # 6. COMBINE ALL SCORES
        total_score = shape_score + color_score + texture_score + edge_score + size_score

        # Decision threshold - if score is high, it's likely a tree
        is_tree = total_score >= 6  # Adjust this threshold as needed

        if is_tree:
            print(f"Tree detected - Shape:{shape_score}, Color:{color_score}, "
                  f"Texture:{texture_score}, Edge:{edge_score}, Size:{size_score}, "
                  f"Total:{total_score}")

        return is_tree

    def count_green_pixels(self, hsv_pixels):
        """Count pixels that fall within green color ranges"""
        if hsv_pixels.size == 0:
            return 0

        green_count = 0
        # Multiple green ranges to catch all types of green
        green_ranges = [
            ([25, 30, 20], [95, 255, 255]),   # Broad green
            ([35, 40, 30], [85, 255, 255]),   # Standard green
            ([45, 50, 40], [75, 255, 255]),   # Specific green
        ]

        for lower, upper in green_ranges:
            lower = np.array(lower)
            upper = np.array(upper)

            # Check each pixel
            for pixel in hsv_pixels:
                if len(pixel) == 3:  # Valid HSV pixel
                    if np.all(pixel >= lower) and np.all(pixel <= upper):
                        green_count += 1
                        break  # Don't double count

        return green_count

    def count_brown_pixels(self, hsv_pixels):
        """Count pixels that fall within brown color ranges (tree trunks)"""
        if hsv_pixels.size == 0:
            return 0

        brown_count = 0
        # Brown color ranges for tree trunks and branches
        brown_ranges = [
            ([8, 50, 30], [25, 200, 150]),    # Standard brown
            ([5, 40, 20], [30, 180, 130]),    # Dark brown
            ([10, 60, 40], [20, 150, 120]),   # Light brown
        ]

        for lower, upper in brown_ranges:
            lower = np.array(lower)
            upper = np.array(upper)

            for pixel in hsv_pixels:
                if len(pixel) == 3:
                    if np.all(pixel >= lower) and np.all(pixel <= upper):
                        brown_count += 1
                        break

        return brown_count

    def filter_building_changes_super_enhanced(self, change_mask, img1, img2, min_area=4000):
        """
        SUPER ENHANCED filtering - removes trees and keeps only building changes
        """
        print("Applying super enhanced filtering with aggressive tree removal...")

        # First, apply aggressive tree change filtering
        tree_filtered_mask = self.filter_tree_changes_aggressively(change_mask, img1, img2)

        # Then apply building shape filtering
        num_labels, labels, stats, centroids = cv2.connectedComponentsWithStats(tree_filtered_mask)
        final_mask = np.zeros_like(tree_filtered_mask)
        valid_building_changes = 0

        for i in range(1, num_labels):
            area = stats[i, cv2.CC_STAT_AREA]

            if area >= min_area:
                component_mask = (labels == i).astype(np.uint8) * 255
                contours, _ = cv2.findContours(
                    component_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
                )

                for contour in contours:
                    if self.is_definitely_building_shape(contour):
                        final_mask[labels == i] = 255
                        valid_building_changes += 1

                        # Additional verification - check if this is really a building change
                        area = cv2.contourArea(contour)
                        x, y, w, h = cv2.boundingRect(contour)
                        print(f"Valid building change {valid_building_changes}: "
                              f"Area={area:.0f}px, Size={w}x{h}px")
                        break

        print(f"Final result: {valid_building_changes} confirmed building changes")
        return final_mask

    def process_images(self, image1_path, image2_path, output_path=None):
        """
        SUPER ENHANCED processing pipeline with aggressive tree filtering
        """
        # Read images
        print(f"Reading images: {image1_path}, {image2_path}")
        img1 = cv2.imread(image1_path)
        img2 = cv2.imread(image2_path)

        if img1 is None or img2 is None:
            raise ValueError("Unable to read image files")

        print("=== SUPER ENHANCED BUILDING CHANGE DETECTION ===")
        print("This version includes aggressive tree filtering")

        # Image registration
        print("Performing image registration...")
        aligned_img2 = self.register_images(img1, img2)

        # Enhanced preprocessing with shadow correction
        print("Preprocessing images with shadow correction...")
        processed_img1 = self.preprocess_image(img1)
        processed_img2 = self.preprocess_image(aligned_img2)

        # Remove shadows first
        print("Detecting and removing shadows...")
        img1_no_shadow, shadow_mask1 = self.detect_and_remove_shadows(processed_img1)
        img2_no_shadow, shadow_mask2 = self.detect_and_remove_shadows(processed_img2)

        # Remove roads
        print("Removing roads...")
        img1_no_road, road_mask1 = self.remove_roads_comprehensive(img1_no_shadow)
        img2_no_road, road_mask2 = self.remove_roads_comprehensive(img2_no_shadow)

        # SUPER ENHANCED vegetation removal
        print("SUPER ENHANCED vegetation removal - aggressively removing all trees...")
        img1_clean, veg_mask1 = self.remove_vegetation_super_enhanced(img1_no_road)
        img2_clean, veg_mask2 = self.remove_vegetation_super_enhanced(img2_no_road)

        # Enhanced building extraction with strict filtering
        print("Extracting building regions with strict shape analysis...")
        building_mask1 = self.extract_building_regions_enhanced(img1_clean)
        building_mask2 = self.extract_building_regions_enhanced(img2_clean)

        # Enhanced building change detection
        print("Detecting building changes with enhanced analysis...")
        change_mask = self.detect_building_changes_enhanced(
            img1_clean, img2_clean, building_mask1, building_mask2
        )

        # Apply morphological operations for building-like shapes
        print("Applying building-specific morphological operations...")
        building_kernel = cv2.getStructuringElement(cv2.MORPH_RECT, (8, 8))
        change_mask = cv2.morphologyEx(
            change_mask,
            cv2.MORPH_OPEN,
            cv2.getStructuringElement(cv2.MORPH_RECT, (5, 5)),
        )
        change_mask = cv2.morphologyEx(change_mask, cv2.MORPH_CLOSE, building_kernel)

        # SUPER ENHANCED filtering with aggressive tree removal
        print("SUPER ENHANCED filtering - aggressively removing tree changes...")
        final_change_mask = self.filter_building_changes_super_enhanced(
            change_mask, img1, img2, min_area=4000
        )

        # Create result visualization
        result_image = self.create_result_image(aligned_img2, final_change_mask)

        # Display results
        self.visualize_results(result_image, output_path)

        # Enhanced statistics
        total_pixels = final_change_mask.shape[0] * final_change_mask.shape[1]
        changed_pixels = np.sum(final_change_mask > 0)
        change_percentage = (changed_pixels / total_pixels) * 100

        print(f"\n=== SUPER ENHANCED BUILDING CHANGE DETECTION RESULTS ===")
        print(f"Total building change pixels: {changed_pixels}")
        print(f"Building change percentage: {change_percentage:.3f}%")

        # Detailed analysis of detected changes
        num_labels, labels = cv2.connectedComponents(final_change_mask)
        building_changes = num_labels - 1

        print(f"Number of confirmed building changes: {building_changes}")
        print("(Tree changes have been aggressively filtered out)")

        # Analyze each detected change
        for i in range(1, num_labels):
            component_mask = (labels == i).astype(np.uint8) * 255
            contours, _ = cv2.findContours(
                component_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
            )

            for contour in contours:
                area = cv2.contourArea(contour)
                x, y, w, h = cv2.boundingRect(contour)
                aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0

                print(f"  Building Change {i}: Area={area:.0f}px, Size={w}x{h}px, "
                      f"Aspect Ratio={aspect_ratio:.2f}")

        return final_change_mask, result_image

    def create_result_image(self, img2, change_mask):
        """
        Create result image with enhanced building change annotations
        """
        result_image = img2.copy()

        # Find contours of change areas
        contours, _ = cv2.findContours(
            change_mask, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE
        )

        # Annotate each building change
        for i, contour in enumerate(contours):
            # Draw thick red outline
            cv2.drawContours(result_image, [contour], -1, (0, 0, 255), 4)

            # Add semi-transparent overlay
            overlay = result_image.copy()
            cv2.fillPoly(overlay, [contour], (0, 100, 255))  # Orange fill
            result_image = cv2.addWeighted(result_image, 0.8, overlay, 0.2, 0)

            # Calculate bounding rectangle and shape metrics
            x, y, w, h = cv2.boundingRect(contour)
            area = cv2.contourArea(contour)

            # Create detailed label
            aspect_ratio = max(w, h) / min(w, h) if min(w, h) > 0 else 0
            label = f"BUILDING Change {i+1}"
            details = f"Area: {area:.0f}px, {w}x{h}px, AR: {aspect_ratio:.1f}"

            # Draw label background
            font_scale = 0.8
            thickness = 2

            # Main label
            label_size = cv2.getTextSize(label, cv2.FONT_HERSHEY_SIMPLEX, font_scale, thickness)[0]
            # Details label
            details_size = cv2.getTextSize(details, cv2.FONT_HERSHEY_SIMPLEX, 0.6, 1)[0]

            # Background rectangle
            max_width = max(label_size[0], details_size[0]) + 20
            total_height = label_size[1] + details_size[1] + 30

            cv2.rectangle(
                result_image,
                (x, y - total_height - 10),
                (x + max_width, y),
                (0, 0, 0),  # Black background
                -1,
            )

            # Draw border
            cv2.rectangle(
                result_image,
                (x, y - total_height - 10),
                (x + max_width, y),
                (0, 0, 255),  # Red border
                2,
            )

            # Draw main label text
            cv2.putText(
                result_image,
                label,
                (x + 10, y - details_size[1] - 15),
                cv2.FONT_HERSHEY_SIMPLEX,
                font_scale,
                (255, 255, 255),  # White text
                thickness,
            )

            # Draw details text
            cv2.putText(
                result_image,
                details,
                (x + 10, y - 5),
                cv2.FONT_HERSHEY_SIMPLEX,
                0.6,
                (200, 200, 200),  # Light gray text
                1,
            )

            # Add center point marker
            center_x = x + w // 2
            center_y = y + h // 2
            cv2.circle(result_image, (center_x, center_y), 5, (0, 255, 0), -1)  # Green dot
            cv2.circle(result_image, (center_x, center_y), 8, (255, 255, 255), 2)  # White circle

        return result_image

    def visualize_results(self, result_image, save_path=None):
        """
        Display the final result with enhanced visualization
        """
        plt.figure(figsize=(16, 12))
        plt.imshow(cv2.cvtColor(result_image, cv2.COLOR_BGR2RGB))
        plt.axis("off")
        plt.title("SUPER ENHANCED Building Change Detection\n(Trees Aggressively Filtered Out)",
                 fontsize=16, fontweight='bold')

        # Add legend
        legend_elements = [
            plt.Rectangle((0, 0), 1, 1, facecolor="red", alpha=0.7, label="Building Changes Only"),
            plt.Circle((0, 0), 1, facecolor="green", label="Change Centers"),
            plt.Rectangle((0, 0), 1, 1, facecolor="orange", alpha=0.3, label="Change Areas"),
        ]
        plt.legend(handles=legend_elements, loc="upper right", fontsize=12)

        if save_path:
            plt.savefig(save_path, dpi=300, bbox_inches="tight")
            print(f"Super enhanced result saved to: {save_path}")

        plt.show()
        return result_image


# Enhanced usage example
def main():
    """
    Main function for SUPER enhanced building-focused change detection
    """
    detector = SuperEnhancedBuildingChangeDetector()

    # Process images
    image1_path = "./images/image_1.jpg"  # Reference image
    image2_path = "./images/image_2.jpg"  # Current image
    output_path = "./images/super_enhanced_building_changes_result.png"

    try:
        change_mask, result_image = detector.process_images(
            image1_path, image2_path, output_path
        )

        # Save results
        if output_path:
            cv2.imwrite("super_enhanced_building_change_mask.png", change_mask)
            cv2.imwrite("super_enhanced_building_changes_annotated.png", result_image)
            print(f"\nSuper enhanced results saved:")
            print(f"- Main result: {output_path}")
            print(f"- Change mask: super_enhanced_building_change_mask.png")
            print(f"- Annotated result: super_enhanced_building_changes_annotated.png")

    except Exception as e:
        print(f"Error during processing: {e}")
        import traceback
        traceback.print_exc()


if __name__ == "__main__":
    main()
